from flask import Flask, render_template, request, redirect, url_for, session, flash, jsonify, Response
from flask_sqlalchemy import SQLAlchemy
from werkzeug.security import generate_password_hash, check_password_hash
from functools import wraps
import os
import secrets
import time
import requests
import json
import re
from threading import Thread
from datetime import datetime

app = Flask(__name__)
app.config['SECRET_KEY'] = os.urandom(32).hex()
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///inkdrop.db'
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False

db = SQLAlchemy(app)

last_report_time = {}
RATE_LIMIT_SECONDS = 30

BOT_PREVIEW_TOKEN = "pk_019c653a34c0718d8afaa7278f0f20ce"
ADMIN_USERNAME = "admin"
ADMIN_PASSWORD = "hunter2"

class User(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password = db.Column(db.String(200), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    posts = db.relationship('Post', backref='author', lazy=True)

class Post(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    title = db.Column(db.String(200), nullable=False)
    content = db.Column(db.Text, nullable=False)
    author_id = db.Column(db.Integer, db.ForeignKey('user.id'), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    is_public = db.Column(db.Boolean, default=True)

def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if 'user_id' not in session:
            flash('Please login first.', 'error')
            return redirect(url_for('login'))
        return f(*args, **kwargs)
    return decorated_function

def render_markdown(content):
    html_content = content
    html_content = re.sub(r'^### (.+)$', r'<h3>\1</h3>', html_content, flags=re.MULTILINE)
    html_content = re.sub(r'^## (.+)$', r'<h2>\1</h2>', html_content, flags=re.MULTILINE)
    html_content = re.sub(r'^# (.+)$', r'<h1>\1</h1>', html_content, flags=re.MULTILINE)
    html_content = re.sub(r'\*\*(.+?)\*\*', r'<strong>\1</strong>', html_content)
    html_content = re.sub(r'\*(.+?)\*', r'<em>\1</em>', html_content)
    html_content = re.sub(r'\[(.+?)\]\((.+?)\)', r'<a href="\2">\1</a>', html_content)
    html_content = html_content.replace('\n\n', '</p><p>')
    html_content = f'<p>{html_content}</p>'
    return html_content

@app.route('/')
def index():
    return render_template('index.html', posts=[])

@app.route('/challenge')
def challenge():
    return render_template('challenge.html', posts=[])

@app.route('/register', methods=['GET', 'POST'])
def register():
    if request.method == 'POST':
        username = request.form.get('username', '').strip()
        password = request.form.get('password', '')
        
        if not username or not password:
            flash('Username and password required.', 'error')
            return render_template('register.html')
        
        if len(username) < 3 or len(username) > 30:
            flash('Username must be 3-30 characters.', 'error')
            return render_template('register.html')
        
        if User.query.filter_by(username=username).first():
            flash('Username exists.', 'error')
            return render_template('register.html')
        
        user = User(username=username, password=generate_password_hash(password))
        db.session.add(user)
        db.session.commit()
        
        flash('Registered! Please login.', 'success')
        return redirect(url_for('login'))
    
    return render_template('register.html')

@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form.get('username', '').strip()
        password = request.form.get('password', '')
        
        user = User.query.filter_by(username=username).first()
        if user and check_password_hash(user.password, password):
            session['user_id'] = user.id
            session['username'] = user.username
            return redirect(url_for('dashboard'))
        
        flash('Invalid credentials.', 'error')
    
    return render_template('login.html')

@app.route('/logout')
def logout():
    session.clear()
    return redirect(url_for('index'))

@app.route('/dashboard')
@login_required
def dashboard():
    user = User.query.get(session['user_id'])
    posts = Post.query.filter_by(author_id=user.id).order_by(Post.created_at.desc()).all()
    return render_template('dashboard.html', posts=posts, user=user)

@app.route('/post/new', methods=['GET', 'POST'])
@login_required
def post_new():
    if request.method == 'POST':
        title = request.form.get('title', '').strip()
        content = request.form.get('content', '').strip()
        
        if not title or not content:
            flash('Title and content required.', 'error')
            return render_template('post_new.html')
        
        post = Post(
            title=title,
            content=content,
            author_id=session['user_id']
        )
        db.session.add(post)
        db.session.commit()
        
        flash('Post created!', 'success')
        return redirect(url_for('post_view', post_id=post.id))
    
    return render_template('post_new.html')

@app.route('/post/<int:post_id>')
def post_view(post_id):
    post = Post.query.get_or_404(post_id)
    
    is_admin = session.get('username') == ADMIN_USERNAME
    is_author = session.get('user_id') == post.author_id
    
    if not is_admin and not is_author:
        flash('Access denied.', 'error')
        return redirect(url_for('index'))
    
    return render_template('post_view.html', post=post)

@app.route('/post/<int:post_id>/edit', methods=['GET', 'POST'])
@login_required
def post_edit(post_id):
    post = Post.query.get_or_404(post_id)
    
    if post.author_id != session['user_id']:
        flash('Access denied.', 'error')
        return redirect(url_for('dashboard'))
    
    if request.method == 'POST':
        post.title = request.form.get('title', '').strip()
        post.content = request.form.get('content', '').strip()
        db.session.commit()
        flash('Post updated!', 'success')
        return redirect(url_for('post_view', post_id=post.id))
    
    return render_template('post_edit.html', post=post)

@app.route('/post/<int:post_id>/delete', methods=['POST'])
@login_required
def post_delete(post_id):
    post = Post.query.get_or_404(post_id)
    if post.author_id != session['user_id']:
        flash('Access denied.', 'error')
        return redirect(url_for('dashboard'))
    
    db.session.delete(post)
    db.session.commit()
    flash('Deleted.', 'success')
    return redirect(url_for('dashboard'))

@app.route('/api/render')
def api_render():
    post_id = request.args.get('id')
    if not post_id:
        return jsonify({'error': 'Missing id'}), 400
    
    post = Post.query.get(post_id)
    if not post:
        return jsonify({'error': 'Not found'}), 404
    
    rendered_html = render_markdown(post.content)
    
    return jsonify({
        'id': post.id,
        'title': post.title,
        'html': rendered_html,
        'author': post.author.username,
        'rendered_at': time.time()
    })

@app.route('/api/jsonp')
def api_jsonp():
    callback = request.args.get('callback', 'handleData')
    
    if '<' in callback or '>' in callback:
        callback = 'handleData'
    
    user_data = {
        'authenticated': 'user_id' in session,
        'timestamp': time.time()
    }
    
    if 'user_id' in session:
        user = User.query.get(session['user_id'])
        if user:
            user_data['username'] = user.username
    
    response = f"{callback}({json.dumps(user_data)})"
    return Response(response, mimetype='application/javascript')

@app.route('/api/config')
def api_config():
    return jsonify({
        'safeMode': True,
        'version': '2.0.0',
        'features': ['markdown', 'preview', 'export']
    })

@app.route('/report/<int:post_id>', methods=['POST'])
@login_required  
def report_post(post_id):
    post = Post.query.get_or_404(post_id)
    user_id = session['user_id']
    current_time = time.time()
    
    if user_id in last_report_time:
        time_diff = current_time - last_report_time[user_id]
        if time_diff < RATE_LIMIT_SECONDS:
            remaining = int(RATE_LIMIT_SECONDS - time_diff)
            flash(f'Wait {remaining}s.', 'error')
            return redirect(url_for('post_view', post_id=post_id))
    
    last_report_time[user_id] = current_time
    
    def send_to_bot(pid):
        try:
            requests.post('http://bot:8090/visit', json={'post_id': pid}, timeout=5)
        except:
            pass
    
    Thread(target=send_to_bot, args=(post_id,)).start()
    flash('Report submitted! Moderator will review.', 'success')
    return redirect(url_for('post_view', post_id=post_id))

@app.route('/health')
def health():
    return 'OK'

with app.app_context():
    db.create_all()
    if not User.query.filter_by(username=ADMIN_USERNAME).first():
        admin = User(username=ADMIN_USERNAME, password=generate_password_hash(ADMIN_PASSWORD))
        db.session.add(admin)
        db.session.commit()

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000, debug=False)
